<?php
// Incluir o arquivo de conexão
try {
    // Note: 'conexao.php' must define $conn as a mysqli object
    include 'conexao.php';
    if (!isset($conn) || !$conn instanceof mysqli) {
        throw new Exception("A variável \$conn não foi definida ou não é uma conexão válida em conexao.php.");
    }
} catch (Exception $e) {
    // Adicionado estilo para erro fatal
    die("<div style='background-color:#fee2e2; color:#991b1b; padding:20px; border:1px solid #fca5a5; font-family:sans-serif;'>Erro fatal de conexão: " . htmlspecialchars($e->getMessage()) . "</div>");
}

// Opcional: Iniciar sessão se este arquivo precisar de autenticação, o que é altamente recomendado em um ambiente de edição de dados RH.
// if (session_status() == PHP_SESSION_NONE) { session_start(); } 
// if (!isset($_SESSION['usuario'])) { header("Location: login.php"); exit; }

$user_id = (int)($_GET['user'] ?? 0);
$funcionario_detalhes = [];
$mensagem_status = '';

// ----------------------------------------------------
// A. TRATAMENTO DO FORMULÁRIO (SALVAR DADOS)
// ----------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $user_id > 0) {
    
    // Lista de colunas a serem atualizadas (ATUALIZADA com todos os 18 campos novos)
    $campos = [
        'bi', 'data_bi', 'categoria', 'salario_liq', 'salario_extenso', 
        'escola', 'iban', 'ano_inicio', 'ano_fim', 'total_anos', 'data_inicio_completa',
        // NOVOS CAMPOS PARA FORÇA DE TRABALHO
        'genero', 'data_nascimento', 'funcao_desempenha', 'situacao_laboral', 
        'disciplina_lecciona', 'formacao_academica', 'data_ingresso_fp', 
        'data_ingresso_educacao', 'inss_inscrito', 'inss_numero', 
        'contacto_unitel', 'contacto_africcel', 'contacto_movicel', 
        'morada_bairro', 'morada_municipio', 'junta_medica_status', 
        'junta_medica_inicio', 'junta_medica_fim'
    ];
    
    // Constrói a query de UPDATE dinamicamente
    $update_parts = [];
    $bind_types = '';
    $bind_values = [];
    
    foreach ($campos as $campo) {
        $update_parts[] = "`{$campo}` = ?";
        $bind_types .= 's'; // Todos os campos são VARCHAR/TEXT, usamos 's' (string)
        $bind_values[] = $_POST[$campo] ?? ''; // Pega o valor do POST
    }
    
    $update_query = "UPDATE usuarios SET " . implode(', ', $update_parts) . " WHERE id = ?";
    
    $bind_types .= 'i'; // Tipo para o ID
    $bind_values[] = $user_id; // Valor para o ID
    
    // Inicia o Prepared Statement
    $stmt = $conn->prepare($update_query);
    
    if ($stmt) {
        // Usa call_user_func_array para passar os parâmetros
        $params = array_merge([$bind_types], $bind_values);
        $refs = [];
        foreach ($params as $key => $value) {
            $refs[$key] = &$params[$key];
        }
        
        call_user_func_array([$stmt, 'bind_param'], $refs);
        
        if ($stmt->execute()) {
            $mensagem_status = "Sucesso: Dados do funcionário (ID {$user_id}) atualizados!";
        } else {
            $mensagem_status = "Erro ao atualizar dados: " . $stmt->error;
        }
        $stmt->close();
    } else {
        $mensagem_status = "Erro ao preparar a query de atualização: " . $conn->error;
    }
}


// ----------------------------------------------------
// B. BUSCA DE DADOS PARA LISTAGEM (BARRA LATERAL)
// ----------------------------------------------------
$funcionarios_listagem = [];
// Selecionar apenas funcionários que não são perfis de sistema (perfil != 'ADMIN' ou 'USER') ou ajustar conforme sua lógica de perfil.
$sql_listagem = "SELECT id, nome, agente FROM usuarios WHERE perfil != 'ADMIN' AND perfil != 'USER' ORDER BY nome ASC"; 
$result_listagem = $conn->query($sql_listagem);
if ($result_listagem) {
    $funcionarios_listagem = $result_listagem->fetch_all(MYSQLI_ASSOC);
}

// ----------------------------------------------------
// C. BUSCA DE DADOS PARA O FORMULÁRIO (SELECIONADO)
// ----------------------------------------------------
if ($user_id > 0) {
    // Usando SELECT * para buscar todos os campos, incluindo os novos
    $sql_detalhes = "SELECT * FROM usuarios WHERE id = ?"; 
    $stmt = $conn->prepare($sql_detalhes);
    if ($stmt) {
        $stmt->bind_param('i', $user_id);
        $stmt->execute();
        $result_detalhes = $stmt->get_result();
        
        if ($result_detalhes && $result_detalhes->num_rows > 0) {
            $funcionario_detalhes = $result_detalhes->fetch_assoc();
        } else {
            $mensagem_status = "Aviso: Detalhes para o funcionário selecionado não encontrados.";
        }
        $stmt->close();
    }
}
?>

<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Editor de Dados RH - SISMAG</title>
    <link rel="stylesheet" href="https://rsms.me/inter/inter.css">
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.2/css/all.min.css">
    <style>
        body {
            font-family: 'Inter', sans-serif;
        }
        /* Cor de foco personalizada para campos de formulário */
        .form-input:focus {
            border-color: #4f46e5; /* indigo-600 */
            box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.5); /* indigo-500 com opacidade */
        }
    </style>
</head>
<body class="bg-gray-100 font-sans">

    <div class="flex h-screen">
        
        <div class="w-80 bg-gray-900 text-white p-6 shadow-2xl flex-shrink-0 overflow-y-auto">
            <h2 class="text-2xl font-extrabold mb-4 text-indigo-400 border-b border-indigo-700 pb-3">
                <i class="fas fa-users-cog mr-2"></i> Editar Dados RH
            </h2>
            
            <a href="index.php" class="flex items-center p-2 rounded-lg text-gray-300 hover:bg-gray-700 hover:text-white transition mb-4">
                <i class="fas fa-arrow-left mr-3"></i> Voltar ao Painel
            </a>
            
            <p class="text-xs text-gray-400 uppercase tracking-wider mb-3 mt-4 border-t border-gray-700 pt-3">Lista de Funcionários</p>
            <ul class="space-y-1">
                <?php if (empty($funcionarios_listagem)): ?>
                    <p class="text-gray-400 text-sm italic">Nenhum funcionário encontrado.</p>
                <?php endif; ?>

                <?php foreach ($funcionarios_listagem as $func): ?>
                    <li class="rounded-lg">
                        <a href="?user=<?= $func['id'] ?>" 
                           class="block p-3 text-sm rounded-md transition duration-200 
                                  <?= $user_id === $func['id'] ? 'bg-indigo-600 font-semibold text-white shadow-md' : 'text-gray-200 hover:bg-gray-700 hover:text-white' ?>">
                            <span class="truncate block"><?= htmlspecialchars($func['nome']) ?></span>
                            <span class="text-xs opacity-75 mt-0.5 block">Agente N.º: <?= htmlspecialchars($func['agente'] ?? 'N/A') ?></span>
                        </a>
                    </li>
                <?php endforeach; ?>
            </ul>
        </div>

        <div class="flex-1 overflow-y-auto p-8 bg-gray-50">
            <h1 class="text-3xl font-bold text-gray-800 mb-8">Editor de Detalhes do Funcionário</h1>
            
            <?php if ($mensagem_status): ?>
                <?php 
                    $is_success = strpos($mensagem_status, 'Sucesso') !== false;
                    $bg_class = $is_success ? 'bg-green-100 border-green-400 text-green-700' : 'bg-red-100 border-red-400 text-red-700';
                    $icon_class = $is_success ? 'fas fa-check-circle' : 'fas fa-exclamation-triangle';
                ?>
                <div class="flex items-center p-4 mb-6 text-sm border-l-4 rounded-lg <?= $bg_class ?>" role="alert">
                    <i class="<?= $icon_class ?> mr-3 text-lg"></i>
                    <p><?= $mensagem_status ?></p>
                </div>
            <?php endif; ?>

            <?php if ($user_id > 0 && !empty($funcionario_detalhes)): ?>
                
                <div class="bg-white p-8 rounded-xl shadow-2xl">
                    <h2 class="text-3xl font-extrabold text-blue-800 mb-6 pb-2 border-b-2 border-indigo-100">
                        <i class="fas fa-address-card mr-2 text-indigo-500"></i>
                        Editando: <?= htmlspecialchars($funcionario_detalhes['nome'] ?? 'N/A') ?>
                        <span class="text-gray-400 text-lg font-normal ml-3">(Agente N.º: <?= htmlspecialchars($funcionario_detalhes['agente'] ?? 'N/A') ?>)</span>
                    </h2>

                    <form method="POST" action="?user=<?= $user_id ?>">
                        <div class="grid grid-cols-1 lg:grid-cols-2 gap-8">

                            <div class="space-y-6">
                                <h3 class="font-bold text-xl border-b pb-3 text-indigo-700"><i class="fas fa-user-circle mr-2"></i> 1. Identificação & Pessoal</h3>

                                <div class="grid grid-cols-1 sm:grid-cols-2 gap-4">
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">BI N.º</span>
                                        <input type="text" name="bi" value="<?= htmlspecialchars($funcionario_detalhes['bi'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border">
                                    </label>

                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Data de Emissão do BI</span>
                                        <input type="text" name="data_bi" value="<?= htmlspecialchars($funcionario_detalhes['data_bi'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: 08/06/2015">
                                    </label>
                                </div>
                                
                                <div class="grid grid-cols-2 gap-4">
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Género</span>
                                        <select name="genero" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border">
                                            <option value="">Selecione</option>
                                            <option value="M" <?= (isset($funcionario_detalhes['genero']) && $funcionario_detalhes['genero'] === 'M') ? 'selected' : '' ?>>Masculino (M)</option>
                                            <option value="F" <?= (isset($funcionario_detalhes['genero']) && $funcionario_detalhes['genero'] === 'F') ? 'selected' : '' ?>>Feminino (F)</option>
                                        </select>
                                    </label>
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Data de Nascimento</span>
                                        <input type="text" name="data_nascimento" value="<?= htmlspecialchars($funcionario_detalhes['data_nascimento'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: 10/05/1980">
                                    </label>
                                </div>
                            </div>

                            <div class="space-y-6">
                                <h3 class="font-bold text-xl border-b pb-3 text-indigo-700"><i class="fas fa-briefcase mr-2"></i> 2. Cargo, Finanças & Formação</h3>
                                
                                <label class="block">
                                    <span class="text-gray-700 font-medium">Categoria/Cargo</span>
                                    <input type="text" name="categoria" value="<?= htmlspecialchars($funcionario_detalhes['categoria'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: Professor do Ensino Primário...">
                                </label>

                                <label class="block">
                                    <span class="text-gray-700 font-medium">Formação Académica</span>
                                    <input type="text" name="formacao_academica" value="<?= htmlspecialchars($funcionario_detalhes['formacao_academica'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: Licenciatura em História">
                                </label>

                                <div class="grid grid-cols-2 gap-4">
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Salário Líquido (AOA)</span>
                                        <input type="text" name="salario_liq" value="<?= htmlspecialchars($funcionario_detalhes['salario_liq'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: 252.695,10">
                                    </label>
                                
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">IBAN N.º</span>
                                        <input type="text" name="iban" value="<?= htmlspecialchars($funcionario_detalhes['iban'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: AO06.0010.0452...">
                                    </label>
                                </div>
                                
                                <label class="block">
                                    <span class="text-gray-700 font-medium">Salário por Extenso (Para Declarações)</span>
                                    <textarea name="salario_extenso" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" rows="2" placeholder="(Duzentos e Cinquenta e Dois mil...)"><?= htmlspecialchars($funcionario_detalhes['salario_extenso'] ?? '') ?></textarea>
                                </label>
                            </div>
                            
                            <div class="space-y-6 lg:col-span-2">
                                <h3 class="font-bold text-xl border-b pb-3 text-indigo-700"><i class="fas fa-calendar-alt mr-2"></i> 3. Local, Funções & Tempo de Serviço</h3>

                                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Escola/Local de Colocação</span>
                                        <input type="text" name="escola" value="<?= htmlspecialchars($funcionario_detalhes['escola'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: Escola Primária BG 1327">
                                    </label>
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Função que Desempenha (Administrativa/Chefia)</span>
                                        <input type="text" name="funcao_desempenha" value="<?= htmlspecialchars($funcionario_detalhes['funcao_desempenha'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: Chefe de Secção, Professor">
                                    </label>
                                </div>
                                
                                <label class="block">
                                    <span class="text-gray-700 font-medium">Disciplina/Classe que Leciona (Apenas Professores)</span>
                                    <input type="text" name="disciplina_lecciona" value="<?= htmlspecialchars($funcionario_detalhes['disciplina_lecciona'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: Matemática, 8ª Classe">
                                </label>

                                <div class="grid grid-cols-3 gap-4">
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Ingresso (Função Pública)</span>
                                        <input type="text" name="data_ingresso_fp" value="<?= htmlspecialchars($funcionario_detalhes['data_ingresso_fp'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: 01/01/2005">
                                    </label>
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Ingresso (Educação)</span>
                                        <input type="text" name="data_ingresso_educacao" value="<?= htmlspecialchars($funcionario_detalhes['data_ingresso_educacao'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: 01/01/2010">
                                    </label>
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Início Completa (Decl. Sub. Morte)</span>
                                        <input type="text" name="data_inicio_completa" value="<?= htmlspecialchars($funcionario_detalhes['data_inicio_completa'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: 01 de Junho de 2015">
                                    </label>
                                </div>
                                
                                <div class="grid grid-cols-3 gap-4 border-t pt-4">
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Ano Início (Contagem)</span>
                                        <input type="text" name="ano_inicio" value="<?= htmlspecialchars($funcionario_detalhes['ano_inicio'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: 2015">
                                    </label>
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Ano Fim (Atual)</span>
                                        <input type="text" name="ano_fim" value="<?= htmlspecialchars($funcionario_detalhes['ano_fim'] ?? date('Y')) ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: 2025">
                                    </label>
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Total Anos (Ex: 10 ou Dez)</span>
                                        <input type="text" name="total_anos" value="<?= htmlspecialchars($funcionario_detalhes['total_anos'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: 10">
                                    </label>
                                </div>
                            </div>

                            <div class="space-y-6">
                                <h3 class="font-bold text-xl border-b pb-3 text-indigo-700"><i class="fas fa-phone-alt mr-2"></i> 4. Situação Laboral & Contacto</h3>

                                <label class="block">
                                    <span class="text-gray-700 font-medium">Situação Laboral</span>
                                    <input type="text" name="situacao_laboral" value="<?= htmlspecialchars($funcionario_detalhes['situacao_laboral'] ?? 'ACTIVO') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: ACTIVO, LICENÇA, JUNTA MÉDICA">
                                </label>

                                <h4 class="font-semibold text-base mt-4 pt-2 border-t">INSS (Instituto Nacional de Segurança Social)</h4>

                                <div class="grid grid-cols-2 gap-4">
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Inscrito no INSS?</span>
                                        <select name="inss_inscrito" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border">
                                            <option value="">Selecione</option>
                                            <option value="SIM" <?= (isset($funcionario_detalhes['inss_inscrito']) && $funcionario_detalhes['inss_inscrito'] === 'SIM') ? 'selected' : '' ?>>SIM</option>
                                            <option value="NÃO" <?= (isset($funcionario_detalhes['inss_inscrito']) && $funcionario_detalhes['inss_inscrito'] === 'NÃO') ? 'selected' : '' ?>>NÃO</option>
                                        </select>
                                    </label>
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">N.º do INSS</span>
                                        <input type="text" name="inss_numero" value="<?= htmlspecialchars($funcionario_detalhes['inss_numero'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border">
                                    </label>
                                </div>
                                
                                <h4 class="font-semibold text-base mt-4 pt-2 border-t">Contactos Telefónicos</h4>

                                <div class="grid grid-cols-3 gap-4">
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Unitel</span>
                                        <input type="text" name="contacto_unitel" value="<?= htmlspecialchars($funcionario_detalhes['contacto_unitel'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border">
                                    </label>
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Africcel</span>
                                        <input type="text" name="contacto_africcel" value="<?= htmlspecialchars($funcionario_detalhes['contacto_africcel'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border">
                                    </label>
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Movicel</span>
                                        <input type="text" name="contacto_movicel" value="<?= htmlspecialchars($funcionario_detalhes['contacto_movicel'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border">
                                    </label>
                                </div>
                            </div>
                            
                            <div class="space-y-6">
                                <h3 class="font-bold text-xl border-b pb-3 text-indigo-700"><i class="fas fa-map-marker-alt mr-2"></i> 5. Morada & Junta Médica</h3>

                                <div class="grid grid-cols-2 gap-4">
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Morada (Bairro/Rua)</span>
                                        <input type="text" name="morada_bairro" value="<?= htmlspecialchars($funcionario_detalhes['morada_bairro'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: Bairro da Caponte">
                                    </label>
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Morada (Município)</span>
                                        <input type="text" name="morada_municipio" value="<?= htmlspecialchars($funcionario_detalhes['morada_municipio'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: Benguela">
                                    </label>
                                </div>
                                
                                <h4 class="font-semibold text-base mt-4 pt-2 border-t">Controlo de Junta Médica</h4>

                                <div class="grid grid-cols-3 gap-4">
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Status Junta Médica</span>
                                        <input type="text" name="junta_medica_status" value="<?= htmlspecialchars($funcionario_detalhes['junta_medica_status'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: Em Avaliação">
                                    </label>
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Início Junta</span>
                                        <input type="text" name="junta_medica_inicio" value="<?= htmlspecialchars($funcionario_detalhes['junta_medica_inicio'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: 01/01/2025">
                                    </label>
                                    <label class="block">
                                        <span class="text-gray-700 font-medium">Fim Junta</span>
                                        <input type="text" name="junta_medica_fim" value="<?= htmlspecialchars($funcionario_detalhes['junta_medica_fim'] ?? '') ?>" class="form-input mt-1 block w-full rounded-md border-gray-300 shadow-sm p-3 border" placeholder="Ex: 01/04/2025">
                                    </label>
                                </div>
                            </div>
                            
                        </div>

                        <div class="mt-10 pt-6 border-t border-gray-200 flex justify-end">
                            <button type="submit" class="bg-indigo-600 text-white px-10 py-3 rounded-full hover:bg-indigo-700 transition duration-300 font-bold shadow-lg transform hover:scale-105">
                                <i class="fas fa-cloud-upload-alt mr-2"></i> Salvar e Atualizar Todos os Dados
                            </button>
                        </div>
                    </form>
                </div>

            <?php else: ?>
                <div class="text-center p-20 bg-white rounded-xl shadow-lg border border-indigo-200">
                    <i class="fas fa-hand-point-left text-6xl text-indigo-400 mb-6 animate-pulse"></i>
                    <h2 class="text-3xl font-semibold text-gray-700">Comece a Edição</h2>
                    <p class="text-gray-500 mt-4 text-lg">Por favor, clique no **nome de um funcionário** na lista à esquerda para carregar e editar seus dados de Recursos Humanos.</p>
                </div>
            <?php endif; ?>

        </div>
    </div>

</body>
</html>